//@author: Martin Zrcek
//@help: Quad with transparent rounded edges
//@tags:
//@credits:

// --------------------------------------------------------------------------------------------------
// PARAMETERS:
// --------------------------------------------------------------------------------------------------

//transforms
float4x4 tW: WORLD;        //the models world matrix
float4x4 tV: VIEW;         //view matrix as set via Renderer (EX9)
float4x4 tP: PROJECTION;
float4x4 tWVP: WORLDVIEWPROJECTION;

//texture
texture Tex <string uiname="Texture Edge";>;
sampler SampEdge = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (Tex);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};
//texture middle
/*texture Tex2 <string uiname="Texture Middle";>;
sampler SampMiddle = sampler_state    //sampler for doing the texture-lookup
{
    Texture   = (Tex2);          //apply a texture to the sampler
    MipFilter = LINEAR;         //sampler states
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};*/

//texture transformation marked with semantic TEXTUREMATRIX to achieve symmetric transformations
float4x4 tTex: TEXTUREMATRIX <string uiname="Texture Transform";>;
float4 colorOut : COLOR <String uiname="Color Glow";>  = {1, 1, 1, 1};
float4 colorIn : COLOR <String uiname="Color Inside";>  = {1, 1, 1, 1};

float2 radius = 0.3;
float2 ranges = 1.0;

//the data structure: "vertexshader to pixelshader"
//used as output data with the VS function
//and as input data with the PS function
struct vs2ps
{
    float4 Pos  : POSITION;
    float2 TexCd : TEXCOORD0;
};

// --------------------------------------------------------------------------------------------------
// VERTEXSHADERS
// --------------------------------------------------------------------------------------------------
vs2ps VS(
    float4 PosO  : POSITION,
    float4 TexCd : TEXCOORD0)
{
    //declare output struct
    vs2ps Out;

    //transform position
    Out.Pos = mul(PosO, tWVP);
    
    //transform texturecoordinates
    Out.TexCd = mul(TexCd, tTex);

    return Out;
}

// --------------------------------------------------------------------------------------------------
// PIXELSHADERS:
// --------------------------------------------------------------------------------------------------

float4 PSLinear(vs2ps In): COLOR
{
    float4 col = tex2D(SampEdge, In.TexCd);
	// Scale Coordinates from (0..1) to (-1..1)
	float2 absCd = abs(In.TexCd-0.5)*2.0;
	absCd = saturate( (absCd-1.0+radius)*(1.0/radius) );
	col.a *= ranges.x;
	absCd = absCd*absCd;
	float add = saturate(sqrt(absCd.x+absCd.y));
	
	col.a -= ranges.y*add;
	if (col.a>=1) return float4(col.rgb*colorIn.rgb,colorIn.a);
	col *= colorOut;
    return saturate(col);
}
float4 PSQuadratic(vs2ps In): COLOR
{
    float4 col = tex2D(SampEdge, In.TexCd);
	float2 absCd = abs(In.TexCd-0.5)*2.0;
	absCd = saturate( (absCd-1.0+radius)*(1.0/radius) );
	
	col.a *= ranges.x;
	float2 value = ranges.y*absCd*absCd;
	if (value.x>0.0) col.a-=value.x;
	if (value.y>0.0) col.a-=value.y;
	if (col.a>=1) return float4(col.rgb*colorIn.rgb,colorIn.a);
	col *= colorOut;
    return saturate(col);
}
float4 PSSmoothStep(vs2ps In): COLOR
{
    float4 col = tex2D(SampEdge, In.TexCd);
	float2 absCd = abs(In.TexCd-0.5)*2.0;
	absCd = saturate( (absCd-1.0+radius)*(1.0/radius) );
	col.a *= ranges.x;
	absCd = absCd*absCd;
	float add = saturate(sqrt(absCd.x+absCd.y));
	
	add = 3*add*add - 2*add*add*add;
	col.a -= ranges.y*add;
	if (col.a>=1) return float4(col.rgb*colorIn.rgb,colorIn.a);
	col *= colorOut;
    return saturate(col);
}

// --------------------------------------------------------------------------------------------------
// TECHNIQUES:
// --------------------------------------------------------------------------------------------------

technique TLinearEdge
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_2_0 PSLinear();
    }
}
technique TQuadraticEdge
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_2_0 PSQuadratic();
    }
}
technique TSmoothStepEdge
{
    pass P0
    {
        //Wrap0 = U;  // useful when mesh is round like a sphere
        VertexShader = compile vs_1_1 VS();
        PixelShader  = compile ps_2_0 PSSmoothStep();
    }
}
